// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/PlayerController.h"

// PNMS Headers
#include "Interfaces/IADMUnifiedControllerPawn.h"
#include "Interfaces/IRiderControllerInterface.h"

// Generated headers
#include "AmsPlayerController.generated.h"

/**
 * Base class for Advanced Mounting System Player Controllers.
 */
UCLASS()
class PNMSADVANCEDSYSTEM_API AAmsPlayerController : public APlayerController, public IADMUnifiedControllerPawn, public IRiderControllerInterface
{
	GENERATED_BODY()

public:
	// Basic Configuration
	////////////////////////////////////////////////////////////////////////////

	/**
	* Setup Default Properties
	*/
	AAmsPlayerController();

public:
	
	/** Set the view target
	 * @param A - new actor to set as view target
	 * @param TransitionParams - parameters to use for controlling the transition
	 */
	virtual void SetViewTarget( class AActor * NewViewTarget, FViewTargetTransitionParams TransitionParams = FViewTargetTransitionParams() ) override;

protected:

	/**
	* Called when spawned or when play begins.
	*/
	virtual void BeginPlay() override;

	/**
	* Called when Destroyed or when Play Ends
	*/
	virtual void EndPlay(const EEndPlayReason::Type endPlayReason) override;

	/**
	 * Overridable native function for when this controller is asked to possess a pawn.
	 * @param InPawn The Pawn to be possessed
	 */
	virtual void OnPossess(APawn* aPawn) override;

	/**
	 * Called when the pawn is unpossessed
	 */
	virtual void OnUnPossess() override;

	/** Called on the client to do local pawn setup after possession, before calling ServerAcknowledgePossession */
	virtual void AcknowledgePossession(APawn * pawn) override;
	
public:

	// Properties
	////////////////////////////////////////////////////////////////////////////

	/**
	* Get the AiPlayer Controller.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="AMS Controller")
	AAmsAIPlayerController * GetAiPlayerController() const { return this->aiPlayerController; }

	/**
	* Called only on clients to set the auto move to enabled or disabled from the pawn
	*/
	void ClientSetAutoMoveMode(bool isAutoMove);

protected:

	// Rider Controller Component
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Replicated, Category="AMS Controller")
	TObjectPtr<class URiderControllerComponent> riderControllerComponent;

	// the AI Player Controller to allow the player to move to position.
	UPROPERTY( VisibleAnywhere, BlueprintReadOnly, Replicated, Category = "AMS Controller" )
	TObjectPtr<class AAmsAIPlayerController> aiPlayerController;

	// The Player AI Class to spawn.
	UPROPERTY( EditDefaultsOnly, BlueprintReadOnly, Category = "AMS Controller" )
	TSubclassOf<class AAmsAIPlayerController> defaultAiPlayerControllerClass;

	// if true the character is entering a period of automatic movement
	UPROPERTY(Transient)
	bool bIsAutoMovementMode = false;

public:

	// Core Functions
	////////////////////////////////////////////////////////////////////////////

	/**
	* Switch the current pawn of this player to use the AI Controller
	*/
	UFUNCTION(BlueprintCallable, Category="AMS Controller")
	void SwitchPawnToAiController(bool bIsClientRequest = false);

	/**
	* Switch the current pawn of this player to use the player controller
	*/
	UFUNCTION(BlueprintCallable, Category="AMS Controller")
	void SwitchPawnToPlayerController(bool bIsClientRequest = false);

	/**
	* Move the player to the specified location
	*/
	UFUNCTION(BlueprintCallable, Category="AMS Controller")
	void AutoMovePlayerToLocation(FVector const & location);

protected:
	
	/**
	* Spawns the player AI Controller to be used for movement later
	*/
	void SpawnPlayerAIController();

	/**
	* Destroys the player AI Controller
	*/
	void DestroyPlayerAIController();

	// Delegate event fired when the pawn has changed
	UFUNCTION()
	virtual void OnPossessedPawnChangedEvent(APawn * oldPawn, APawn * newPawn);

public:

	// IAdmUnifiedControllerPawn
	////////////////////////////////////////////////////////////////////////////

	/** 
	* Get the Character's Controller
	* This will only work on the owning client or the server
	* @return class AController *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IADMUnifiedControllerPawn")
	class AController * GetCharacterController() const;
	virtual class AController * GetCharacterController_Implementation() const override;

	/** 
	* Get the Character that the player or bot is represented as
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class APawn * GetCharacterPawn() const;
	virtual class APawn * GetCharacterPawn_Implementation() const override;

	/**
	* Get the Mount that the character is riding, null if no mount.
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class AActor * GetCharacterMount() const;
	virtual class AActor * GetCharacterMount_Implementation() const override;

	/** 
	* Get flag indicating that the character is mounted.
	* this works for either the Rider or the Mount itself
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool IsMounted() const;
	virtual bool IsMounted_Implementation() const override;

	/**
	* Called to Prepare the Pawn or Controller for Mounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToMount(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response);
	virtual bool PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response) override;

	/**
	* Prepare the Pawn or Controller for Dismounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToDismount(AActor * mountOrRider, FMountActionResponse & response);
	virtual bool PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response) override;

public:

	// IRiderControllerInterface
	////////////////////////////////////////////////////////////////////////////

	/**
	* Begins the Mounting Process
	* @param newMount - Actor to mount to
	* @param linkedActor - Linked Actor to mount to
	* @param seatId - seat to mount to
	* @param mountingResponse - Status Response of the mounting if it succeeded.
	* @return bool - true if the mounting operation succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginMountingActor(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse);
	virtual bool BeginMountingActor_Implementation(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse) override;

	/**
	* Begins the Mounting Process
	* @param newMount - Actor to mount to
	* @param linkedActor - Linked Actor to mount to
	* @param seatId - seat to mount to
	* @param mountingResponse - Status Response of the mounting if it succeeded.
	* @return bool - true if the mounting operation succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginMountingActorToSeat(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse);
	virtual bool BeginMountingActorToSeat_Implementation(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse) override;

	/** 
	* Starts the dismounting action
	* @param response - output status response if mounting succeeded or failed
	* @return bool - true if dismount succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginDismountingActor(FMountActionResponse & response);
	virtual bool BeginDismountingActor_Implementation(FMountActionResponse & response) override;

	/**
	* Begin changing seats to the specified seat ID.
	* @param seatId - ID of the seat to change to
	* @param mountingResponse - status Response of the seat change
	* @return bool - true if the seat change was successful
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginChangingSeatById(int32 seatId, struct FMountActionResponse & mountingResponse);
	virtual bool BeginChangingSeatById_Implementation(int32 seatId, struct FMountActionResponse & mountingResponse) override;

	/**
	* Begin changing seats to the specified seat index.
	* @param seatIndex - index of the seat to change to
	* @param mountingResponse - status Response of the seat change
	* @return bool - true if the seat change was successful
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginChangingSeatToIndex(int32 seatIndex, struct FMountActionResponse & mountingResponse);
	virtual bool BeginChangingSeatToIndex_Implementation(int32 seatIndex, struct FMountActionResponse & mountingResponse) override;

	/** 
	* Get the Rider Controller Component
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	class URiderControllerComponent * GetRiderControllerComponent() const;
	virtual class URiderControllerComponent * GetRiderControllerComponent_Implementation() const override;

	/** 
	* Called to tell the controller to possess the given Pawn
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController", meta=(DisplayName="Possess Pawn (PNMS)"))
	bool PNMS_PossessPawn(APawn * pawnToPossess);
	virtual bool PNMS_PossessPawn_Implementation(APawn * pawnToPossess) override;

	/** 
	* Called when a new rider is added to the current mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnRiderAdded(AActor * newRider, int32 seatId);
	virtual bool OnRiderAdded_Implementation(AActor * newRider, int32 seatId) override;

	/**
	* Called when a rider is removed from the current mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnRiderRemoved(AActor * removedRider, int32 seatId);
	virtual bool OnRiderRemoved_Implementation(AActor * removedRider, int32 seatId) override;

	/**
	* Called when another rider changes to a new seat on the mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnOtherRiderChangedSeats(AActor * otherRider, int32 newSeatId, int32 oldSeatId);
	virtual bool OnOtherRiderChangedSeats_Implementation(AActor * otherRider, int32 newSeatId, int32 oldSeatId) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to teh failure of the mounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsMountingActionFailed(FPnmsActionEventArgs const & pnmsActionResultArgs);
	virtual bool OnPnmsMountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsDisountingActionFailed(FPnmsActionEventArgs const & pnmsActionResultArgs);
	virtual bool OnPnmsDisountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsChangeSeatActionFailed(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs);
	virtual bool OnPnmsChangeSeatActionFailed_Implementation(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs) override;

public:

	// Network and RPCs
	////////////////////////////////////////////////////////////////////////////

	/**
	* Setup the replicated properties for this class
	*/
	virtual void GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const override;


	/**
	* Called to switch pawn to AI Controller
	*/
	UFUNCTION(Server, WithValidation, Reliable)
	void SERVER_SwitchPawnToAiController();
	void SERVER_SwitchPawnToAiController_Implementation();
	bool SERVER_SwitchPawnToAiController_Validate();

	/**
	* Called to switch the pawn to the Player Controller
	*/
	UFUNCTION(Server, WithValidation, Reliable)
	void SERVER_SwitchPawnToPlayerController();
	void SERVER_SwitchPawnToPlayerController_Implementation();
	bool SERVER_SwitchPawnToPlayerController_Validate();

	// Called to start the auto move system on the server
	UFUNCTION(Server, WithValidation, Reliable)
	void SERVER_AutoMovePlayerToLocation(FVector const & location);
	void SERVER_AutoMovePlayerToLocation_Implementation( FVector const & location );
	bool SERVER_AutoMovePlayerToLocation_Validate( FVector const & location );

	// In the event there is an error on the server when enabling and disabling automatic movement, inform the client to reset
	UFUNCTION(Client, Reliable)
	void CLIENT_ResetAutoMoveFlag(bool bAutoMoveFlag);
	void CLIENT_ResetAutoMoveFlag_Implementation( bool bAutoMoveFlag );
}; 
