// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#pragma once

// Engine Includes
#include "CoreMinimal.h"
#include "GameFramework/Character.h"
#include "Types/AmsTypes.h"

// PNMS Includes
#include "Interfaces/IADMUnifiedControllerPawn.h"
#include "Interfaces/IMountablePawnInterface.h"

// AMS Includes
#include "Interfaces/IAmsMountableActor.h"

// Generated Includes
#include "PnmsAdvancedMountCharacterBase.generated.h"

/**
* This is the base class for Mounts based on the Character Class, this does not include playable characters
*/
UCLASS()
class PNMSADVANCEDSYSTEM_API APnmsAdvancedMountCharacterBase : public ACharacter, public IADMUnifiedControllerPawn, public IMountablePawnInterface, public IAmsMountableActorInterface
{
	GENERATED_BODY()

public:
	// Basic Configuration
	///////////////////////////////////////////////////////////////////////////////////
	
	/**
	* Setup Default values
	*/
	APnmsAdvancedMountCharacterBase();

	/**
	* Called when the Pawn is possessed by a controller
	* @param NewController - new AController possessing this character
	*/
	virtual void PossessedBy(AController* NewController) override;

public:
	// Properties
	///////////////////////////////////////////////////////////////////////////////////
	
protected:

	// Reference to the AI Controller that this mount was given.
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Replicated, Category="PNMS Mount Character")
	TObjectPtr<class AController> MountAiController;

	// Component that is responsible for Mount Management
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Replicated, Category="PNMS Mount Character")
	TObjectPtr<class UMountablePawnComponent> mountablePawnComponent;

	// Defines the type of mount this represents
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, Replicated, Category="PNMS Mount Character")
	EAmsMountTypes mountType = EAmsMountTypes::Invalid;

public:
	// IAdmUnifiedControllerPawn
	///////////////////////////////////////////////////////////////////////////////////

	/** 
	* Get the Character's Controller
	* This will only work on the owning client or the server
	* @return class AController *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IADMUnifiedControllerPawn")
	class AController * GetCharacterController() const;
	virtual class AController * GetCharacterController_Implementation() const override;

	/** 
	* Get the Character that the player or bot is represented as
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class APawn * GetCharacterPawn() const;
	virtual class APawn * GetCharacterPawn_Implementation() const override;

	/**
	* Get the Mount that the character is riding, null if no mount.
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class AActor * GetCharacterMount() const;
	virtual class AActor * GetCharacterMount_Implementation() const override;

	/** 
	* Get flag indicating that the character is mounted.
	* this works for either the Rider or the Mount itself
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool IsMounted() const;
	virtual bool IsMounted_Implementation() const override;

	/**
	* Called to Prepare the Pawn or Controller for Mounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToMount(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response);
	virtual bool PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response) override;

	/**
	* Prepare the Pawn or Controller for Dismounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToDismount(AActor * mountOrRider, FMountActionResponse & response);
	virtual bool PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response) override;

public:

	// IMountablePawnInterface
	////////////////////////////////////////////////////////////////////////////

	/** 
	* Get the Mountable Pawn Component
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	class UMountablePawnComponent * GetMountablePawnComponent() const;
	virtual class UMountablePawnComponent * GetMountablePawnComponent_Implementation() const override;

	/**
	* Get flag indicating that the Pawn Is a Mountable Actor
	* Note that the IADMUnifiedControllerPawn has a similar function ICharacterMountable that should return the same value
	* @return bool - true if mountable
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsMountableActor() const;
	virtual bool IsMountableActor_Implementation() const override;

	/**
	* Get flag indicating that the Pawn can currently be mounted, e.g. it has no rider or has an open seat.
	* @param newRider - rider wanting to mount
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool CanMountActor(AActor * newRider) const;
	virtual bool CanMountActor_Implementation(AActor * newRider) const override;

	/**
	* Get flag indicating that the Pawn currently has any number of passengers (or a driver) currently on it
	* @return bool
	* @Version 6
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool HasPassengers() const;
	virtual bool HasPassengers_Implementation() const override;

	/**
	* Get flag indicating that the supplied pawn is allowed to mount or enter the vehicle
	* @param newRider - APawn that wants to mount
	* @return bool - true if allowed
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsMountableByPawn(APawn * newRider) const;
	virtual bool IsMountableByPawn_Implementation(APawn * newRider) const override;

	/** 
	* Get flag indicating that the pawn has a current driver
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool HasDriver() const;
	virtual bool HasDriver_Implementation() const override;

	/**
	* Get flag indicating that the Driver is ready to take control and drive the vehicle or mount
	* @bool
	* @Version 6
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsDriverReady() const;
	virtual bool IsDriverReady_Implementation() const override;

	/** 
	* Flag indicating that this mount must have a driver before it can be ridden
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool MustHaveDriver() const;
	virtual bool MustHaveDriver_Implementation() const override;

	/** 
	* Get the Driver of this Mount
	* @return APawn
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	APawn * GetDriver() const;
	virtual APawn * GetDriver_Implementation() const override;

	/** 
	* Determine if the given seat is the driver seat of the mount.  Must be a possessable mount to have a driver seat.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsDriverSeat(const FSeatData & seatData) const;
	virtual bool IsDriverSeat_Implementation(const FSeatData & seatData) const override;

	/**
	* Get the maximum number of Riders the mount can support
	* return int32
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	int32 GetMaxRiders() const;
	virtual int32 GetMaxRiders_Implementation() const override;

	/**
	* Get the current number of Riders the mount has
	* @return int32
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	int32 GetCurrentRiderCount() const;
	virtual int32 GetCurrentRiderCount_Implementation() const override;

	/**
	* Find the nearest available mounting seat on the mount relatifve to the rider and desired mounting position
	* @param position - EMountingPosition position the rider desires to mount
	* @param riderLocation - FVector location the rider is currently standing to mount
	* @param outSeatData - FMountSeatData to mount the rider to
	* @param outSeatIndex - index of the seat to attach to
	* @return bool - true if a seat was found
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool FindAvailableMountingPosition(EMountingDirection position, FVector riderLocation, FSeatData & outSeatData, int32 & outSeatIndex) const;
	virtual bool FindAvailableMountingPosition_Implementation(EMountingDirection position, FVector riderLocation, FSeatData & outSeatData, int32 & outSeatIndex) const override;

	/**
	* Get flag indicating that the seat at the specified index is occupied
	* @param seatIndex - int32 index of the seat to check
	* @return bool - true is the seat is occupied or the index is invalid
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsSeatOccupiedAtIndex(int32 seatIndex) const;
	virtual bool IsSeatOccupiedAtIndex_Implementation(int32 seatIndex) const override;

	/**
	* Clear the specified Seat of being occupied
	* @param seatIndex - index of the seat to clear
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool IsSeatOccupiedById(int32 seatId) const;
	virtual bool IsSeatOccupiedById_Implementation(int32 seatId) const override;

	/**
	* Set the seat at the specified index as occupied by the rider
	* @param seatIndex - int32 index of the seat to set occupied
	* @param rider - APawn to set as the rider
	* @return bool - true if operaiton succeeds
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool SetSeatOccupiedAtIndex(int32 seatIndex, APawn * rider);
	virtual bool SetSeatOccupiedAtIndex_Implementation(int32 seatIndex, APawn * rider) override;

	/**
	* Set the seat by the specified id as occupied by the rider
	* @param seatIndex - int32 id of the seat to set occupied
	* @param rider - APawn to set as the rider
	* @return bool - true if operation succeeds
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool SetSeatOccupiedById(int32 seatId, APawn * rider);
	virtual bool SetSeatOccupiedById_Implementation(int32 seatId, APawn * rider) override;

	/**
	* Clear the specified Seat of being occupied
	* @param seatIndex - index of the seat to clear
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool ClearSeatAtIndex(int32 seatIndex);
	virtual bool ClearSeatAtIndex_Implementation(int32 seatIndex) override;

	/**
	* Clear the specified seat from being occupied
	* @param seatIndex - int32 index of the seat to clear
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool ClearSeatById(int32 seatId);
	virtual bool ClearSeatById_Implementation(int32 seatId) override;

	/**
	* Get data about the seat at the specified index
	* @param index - int32 index of the seat to retrieve
	* @param seatData - will hold the data of the seat in question
	* @return bool - true if the seat was found
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool GetSeatDataAtIndex(int32 index, FSeatData & seatData) const;
	virtual bool GetSeatDataAtIndex_Implementation(int32 index, FSeatData & seatData) const override;

	/**
	* Get data about the seat by the specified id
	* @param seatId - int32 id of the seat to retrieve
	* @param seatData - will hold the data of the seat in question
	* @return bool - true if the seat was found
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool GetSeatDataById(int32 seatId, FSeatData & seatData) const;
	virtual bool GetSeatDataById_Implementation(int32 seatId, FSeatData & seatData) const override;

	/**
	* Get flag indicating that a rider can mount given the desired mounting position and their current location.
	* @param riderLocation - FVector
	* @param desiredMountingPosition - EMountingPosition
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool CanMountAtPosition(FVector riderLocation, EMountingDirection desiredMountingPosition) const;
	virtual bool CanMountAtPosition_Implementation(FVector riderLocation, EMountingDirection desiredMountingPosition) const override;

	/**
	* Retrieves a valid relative mounting position for the mount according to where the rider is currently standing
	* This function allows for a more forgiving location than the one in the Tools Library.
	* @param rider - rider actor.
	* @todo - Change to GetRelativeDirectionToMount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	EMountingDirection GetRelativeMountDirection(APawn * rider) const;
	virtual EMountingDirection GetRelativeMountDirection_Implementation(APawn * rider) const override;

	/**
	* Get direction to dismount from.
	* @param rider - AActor
	* @todo - Change to GetRelativeDirectionToDismount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	EMountingDirection GetRelativeDismountDirection(APawn * rider) const;
	virtual EMountingDirection GetRelativeDismountDirection_Implementation(APawn * rider) const override;

	/**
	* Get the Mesh Component to mount the character to.
	* @param seatId - id of the seat to retrieve the body for
	* @return UMeshComponent
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	UMeshComponent * GetMountBody(int32 seatId) const;
	virtual UMeshComponent * GetMountBody_Implementation(int32 seatId) const override;

	/** 
	* Called when a rider completes their mounting procedure signaling they have occupied a seat
	* @param mountedActor - actor that is now mounted
	* @param seatId - id of the seat the actor has taken
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool OnRiderFinishedMounting(AActor * mountedActor, int32 seatId);
	virtual bool OnRiderFinishedMounting_Implementation(AActor * mountedActor, int32 seatId) override;

	/**
	* Called when a rider completes their dismounting procedure signaling they have freed up a seat
	* @param dismountedActor - actor that has left their seat
	* @param seatId - id of the seat the actor has left
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool OnRiderFinishedDismounting(AActor * dismountedActor, int32 seatId);
	virtual bool OnRiderFinishedDismounting_Implementation(AActor * dismountedActor, int32 seatId) override;

	/**
	* Called when a rider finished moving from one seat to another signaling that a former seat has been freed up
	* @param rider - rider who changed seats
	* @param newSeatId - new seat the rider is occupying
	* @param oldSeatId - old seat the rider has left
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountablePawn")
	bool OnRiderFinishedChangingSeats(AActor * rider, int32 newSeatId, int32 oldSeatId);
	virtual bool OnRiderFinishedChangingSeats_Implementation(AActor * rider, int32 newSeatId, int32 oldSeatId) override;

public:

	// IAmsMountableActor Interface
	////////////////////////////////////////////////////////////////////////////

	/**
	* Get the type of Mount the implementing actor is considered.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountableActor")
	EAmsMountTypes GetMountType() const;
	virtual EAmsMountTypes GetMountType_Implementation() const override;

	/**
	* Called when the the mount driver dismounts
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IAmsMountableActor")
	void OnMountDriverDismounted();
	virtual void OnMountDriverDismounted_Implementation() override;

public:

	// Network and RPCs
	////////////////////////////////////////////////////////////////////////////

	/**
	* Setup the replicated properties for this class
	*/
	virtual void GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const override;
};


