// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#pragma once

// Engine Includes
#include "CoreMinimal.h"
#include "GameFramework/Character.h"
#include "AbilitySystemInterface.h"
#include "GameplayCueInterface.h"
#include "GameplayTagAssetInterface.h"

// PNMS Headers
#include "Interfaces/IADMUnifiedControllerPawn.h"
#include "Interfaces/IMountRiderInterface.h"

// Ams Includes
#include "Interfaces/IAmsMountRider.h"

// Generated Headers
#include "AmsRiderCharacter.generated.h"

/**
* Base class for Rider Character that can be inherited in Blueprints or other C++
*/
UCLASS()
class PNMSADVANCEDSYSTEM_API AAmsRiderCharacter : public ACharacter, public IADMUnifiedControllerPawn, public IMountRiderInterface, public IAbilitySystemInterface, public IAmsMountRiderInterface
{
	GENERATED_BODY()

public:
	// Basic Configuration
	///////////////////////////////////////////////////////////////////////////////////
	
	/**
	* Sets default values for this character's properties
	*/
	AAmsRiderCharacter();

	/**
	* Called when the character is possessed by a new Controller on the server
	*/
	virtual void PossessedBy(AController * newController) override;

public:

	// Properties
	///////////////////////////////////////////////////////////////////////////////////

protected:

	// Rider Component that handles settings for the Rider Character
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Replicated, Category="Ams Rider Character")
	TObjectPtr<class UMountRiderComponent> mountRiderComponent;

	// ability system component
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Replicated, Category="Ams Rider Character")
	TObjectPtr<class UAbilitySystemComponent> abilitySystemComponent;

	// The minimum distance on the XY plane between the mount and the player for a move to be needed.
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Ams Rider Character")
	float minMountDistanceMoveThreshold = 180.0f;

	// Reference to Mounting Data for this class
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, Category="Ams Animation Data")
	UAmsMountingAnimationData * mountingAnimationData = nullptr;

	// flag indicating that the automove system is enabled or disabled
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, ReplicatedUsing=OnRep_IsAutomoveEnabled, Category="Ams Animation Data")
	bool isAutomoveEnabled = false;

public:

	// Methods
	///////////////////////////////////////////////////////////////////////////////////

protected:

	/**
	* Called to begin the small wait timer for the switch over to the player controller
	*/
	void StartMountingLocaitonCompleteTimer(float duration = 1.0f);

	/**
	* Callback for timer to start the move to location
	*/
	UFUNCTION()
	void OnMoveToLocationCompleted();

public:

	// IAdmUnifiedControllerPawn
	///////////////////////////////////////////////////////////////////////////////////

	/** 
	* Get the Character's Controller
	* This will only work on the owning client or the server
	* @return class AController *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IADMUnifiedControllerPawn")
	class AController * GetCharacterController() const;
	virtual class AController * GetCharacterController_Implementation() const override;

	/** 
	* Get the Character that the player or bot is represented as
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class APawn * GetCharacterPawn() const;
	virtual class APawn * GetCharacterPawn_Implementation() const override;

	/**
	* Get the Mount that the character is riding, null if no mount.
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class AActor * GetCharacterMount() const;
	virtual class AActor * GetCharacterMount_Implementation() const override;

	/** 
	* Get flag indicating that the character is mounted.
	* this works for either the Rider or the Mount itself
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool IsMounted() const;
	virtual bool IsMounted_Implementation() const override;

	/**
	* Called to Prepare the Pawn or Controller for Mounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToMount(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response);
	virtual bool PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response) override;

	/**
	* Prepare the Pawn or Controller for Dismounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToDismount(AActor * mountOrRider, FMountActionResponse & response);
	virtual bool PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response) override;

public:

	// IMountRiderInterface
	///////////////////////////////////////////////////////////////////////////////////

	/**
	* Get the Mount Rider Component
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	class UMountRiderComponent * GetMountRiderComponent() const;
	virtual class UMountRiderComponent * GetMountRiderComponent_Implementation() const override;

	/**
	* Starts mounting process for given mounted pawn
	* @param newMountActor - Actor to mount
	* @param linkedActor - optional actor with a link to the mount
	* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the mounting
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool StartPawnMounting(class AActor * newMountActor, AActor * linkedActor, struct FMountActionResponse & mountingResponse);
	virtual bool StartPawnMounting_Implementation(class AActor * newMountActor, AActor * linkedActor, struct FMountActionResponse & mountingResponse) override;

	/**
	* Starts mounting process for given mounted pawn
	* @param newMountActor - Actor to mount
	* @param linkedActor - optional actor with a link to the mount
	* @param seatId - seat to attach to
	* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the mounting
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool StartPawnMountingToSeat(class AActor * newMountActor, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse);
	virtual bool StartPawnMountingToSeat_Implementation(class AActor * newMountActor, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse) override;

	/**
	* Called when the mounting process has completed
	* @param newMountActor - Actor to mount
	* @param linkedActor - optional actor with a link to the mount
	* @return bool - true in order to treat as function in blueprint instead of event
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool OnMountingPawnFinished(class AActor * newMountActor);
	virtual bool OnMountingPawnFinished_Implementation(class AActor * newMountActor) override;

	/**
	* Starts dismounting process for given mounted pawn
	* @param oldPawnMount - Actor to dismount
	* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the dismounting
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool StartPawnDismounting(class AActor * oldPawnMount, struct FMountActionResponse & mountingResponse);
	virtual bool StartPawnDismounting_Implementation(class AActor * oldPawnMount, struct FMountActionResponse & mountingResponse) override;

	/**
	* Called when the mounting process has completed
	* @param dismountedPawn - APawn being dismounted
	* @return bool - true in order to treat as function in blueprint instead of event
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool OnDismountingPawnFinished(class AActor * oldPawnMount);
	virtual bool OnDismountingPawnFinished_Implementation(class AActor * oldPawnMount) override;

	/**
	* Get flag indicating that the rider is the driver of the mount/vehicle
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool IsDriver() const;
	virtual bool IsDriver_Implementation() const override;

	/**
	* Get flag indicating that the rider is seated on the mount.
	* @return bool
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool IsSeatedOnMount() const;
	virtual bool IsSeatedOnMount_Implementation() const override;

	/** 
	* Get the ID of the seat the rider is currently mounted on.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	int32 GetSeatId() const;
	virtual int32 GetSeatId_Implementation() const override;

	/**
	* Allow movement to the specified location
	* @return bool - true if teleporting to that location, false if movement to location is required
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool MoveToMountingLocation(const FVector & location, const FRotator & orientation);
	virtual bool MoveToMountingLocation_Implementation(const FVector & location, const FRotator & orientation) override;

	/**
	* Called when moving the the desired mounting location is completed
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool OnMoveToMountingLocationCompleted();
	virtual bool OnMoveToMountingLocationCompleted_Implementation() override;

	/**
	* Called when player has finished changing to their new seat
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool OnChangeToNewSeatCompleted();
	virtual bool OnChangeToNewSeatCompleted_Implementation() override;

	/**
	* Play mounting animation based on current position
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider|Animation")
	bool PlayMountingAnimation(EMountingDirection position);
	virtual bool PlayMountingAnimation_Implementation(EMountingDirection position) override;

	/**
	* Play mounting animation based on current position
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider|Animation")
	bool PlayDismountingAnimation(EMountingDirection position);
	virtual bool PlayDismountingAnimation_Implementation(EMountingDirection position) override;

	/**
	* Play animation for moving character into their new seat.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider|Animation")
	bool PlayMoveToSeatAnimation(int32 currentSeatId, int32 oldSeatId);
	virtual bool PlayMoveToSeatAnimation_Implementation(int32 currentSeatId, int32 oldSeatId) override;

	/**
	* Get the Mesh component of the Rider
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	UMeshComponent * GetRiderMesh() const;
	virtual UMeshComponent * GetRiderMesh_Implementation() const override;

	/**
	* Enable or Disable the collision of a Rider
	* @param collisionType - new collision type for the rider
	* @return bool - true if collision changes where handled by this function
	* @Version 6
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool SetRiderCollisionEnabled(bool shouldEnable);
	virtual bool SetRiderCollisionEnabled_Implementation(bool shouldEnable) override;

	/**
	* Updates the rider movement mode for when the character is mounted or not
	* @param isMounted - bool flag to indicate if we are mounted or not
	* @param mountActor - actor that is currently our mount.
	* @return bool - true if we handled the update.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	bool UpdateRiderMovementMode(const bool isMounted, const AActor * mountActor);
	virtual bool UpdateRiderMovementMode_Implementation(const bool isMounted, const AActor * mountActor) override;

	/**
	* Retrieve all Skeletal Mesh Objects that make up a Rider which need to Animate on a Server.
	* This is a bug fix for an issue which occurs when turning Replicate Movement off on both Listen and Dedicated Servers
	* which stops all animation updates on the server.
	* @return TArray<USkeletalMeshComponent*>
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IMountRider")
	TArray<class USkeletalMeshComponent*> GetAllRiderSkeletalMeshes() const;
	virtual TArray<class USkeletalMeshComponent*> GetAllRiderSkeletalMeshes_Implementation() const override;

	// IAmsMountRider Interface
	///////////////////////////////////////////////////////////////////////////////////

	/**
	* Gets the Mounting Animation Data Objiect for the Rider
	* @return UAmsMountingAnimationData *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAmsMountingAnimationData * GetMountingAnimationData() const;
	virtual class UAmsMountingAnimationData * GetMountingAnimationData_Implementation() const override;

	/**
	* Find the AnimMontage for mounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occuring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForMounting(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const;
	virtual class UAnimMontage * GetMontageForMounting_Implementation(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const override;

	/**
	* Find the AnimMontage for dismounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occuring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForDismounting(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const;
	virtual class UAnimMontage * GetMontageForDismounting_Implementation(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const override;

	/**
	* Get the AnimMontage for changing from one seat to another.
	* @param mountType - type of mount being used
	* @param startSeatName - name of the seat the change is starting from
	* @param destinationSeatName - name of the seat to end at
	* @return UAnimMontage to play
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForChangingSeatsByName(EAmsMountTypes const mountType, FName const & startingSeatName, FName const & destinationSeatName) const;
	virtual class UAnimMontage * GetMontageForChangingSeatsByName_Implementation(EAmsMountTypes const mountType, FName const & startingSeatName, FName const & destinationSeatName) const override;

	/**
	* Get the AnimMontage for changing from one seat to another.
	* @param mountType - type of mount being used
	* @param startSeatName - name of the seat the change is starting from
	* @param destinationSeatName - name of the seat to end at
	* @return UAnimMontage to play
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForChangingSeatsById(EAmsMountTypes const mountType, int32 const startingSeatId, int32 const destinationSeatId) const;
	virtual class UAnimMontage * GetMontageForChangingSeatsById_Implementation(EAmsMountTypes const mountType, int32 const startingSeatId, int32 const destinationSeatId) const override;

	/**
	* Enables and Disabled the auto move functionality on the rider
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	bool SetAutoMoveEnabled(bool isEnabled);
	virtual bool SetAutoMoveEnabled_Implementation(bool isEnabled);

public:

	// IAbilitySystemInterface
	///////////////////////////////////////////////////////////////////////////////////

	/** Returns the ability system component to use for this actor. It may live on another actor, such as a Pawn using the PlayerState's component */
	virtual UAbilitySystemComponent* GetAbilitySystemComponent() const override;

public:

	// Network and RPCs
	////////////////////////////////////////////////////////////////////////////

	/**
	* Setup the replicated properties for this class
	*/
	virtual void GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const override;

	/**
	* Called when auto move enable and disable is replicated
	*/
	UFUNCTION()
	virtual void OnRep_IsAutomoveEnabled(bool wasAutomoveEnabled);
};
