// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/


#include "Player/AmsPlayerController.h"
#include "Components/RiderControllerComponent.h"

#include "AbilitySystemInterface.h"
#include "AbilitySystemComponent.h"
#include "AI/AmsAIPlayerController.h"

#include "Pawn/AmsRiderCharacter.h"
#include "PnmsAdvancedSystemPublicPCH.h"

#include "Interfaces/IMountRiderInterface.h"
#include "Interfaces/IMountablePawnInterface.h"
#include "Components/MountRiderComponent.h"


// Basic Configuration
////////////////////////////////////////////////////////////////////////////

/**
* Setup Default Properties
*/
AAmsPlayerController::AAmsPlayerController()
{
	this->riderControllerComponent = CreateDefaultSubobject<URiderControllerComponent>("RiderControllerComponent");
	this->riderControllerComponent->SetEnableActionRpcsByDefault(true);	

	this->defaultAiPlayerControllerClass = AAmsAIPlayerController::StaticClass();
}

/** Set the view target
* @param A - new actor to set as view target
* @param TransitionParams - parameters to use for controlling the transition
*/
void AAmsPlayerController::SetViewTarget( class AActor * NewViewTarget, FViewTargetTransitionParams TransitionParams )
{
	if ( !this->bIsAutoMovementMode )
	{
		Super::SetViewTarget(NewViewTarget, TransitionParams);
	}
}

/**
* Called when spawned or when play begins.
*/
void AAmsPlayerController::BeginPlay()
{
	Super::BeginPlay();

	// Register with the delegate which is fired when the possessed pawn is changed.
	this->OnPossessedPawnChanged.AddUniqueDynamic(this, &AAmsPlayerController::OnPossessedPawnChangedEvent);

	// If this is the server/authority spawn the special Player AI Controller
	if ( this->HasAuthority() )
	{
		this->SpawnPlayerAIController();
	}
}

/**
* Called when Destroyed or when Play Ends
*/
void AAmsPlayerController::EndPlay( const EEndPlayReason::Type endPlayReason )
{
	this->OnPossessedPawnChanged.RemoveDynamic( this, &AAmsPlayerController::OnPossessedPawnChangedEvent );
	if ( this->HasAuthority() )
	{
		this->DestroyPlayerAIController();
	}

	Super::EndPlay(endPlayReason);
}

/**
* Over ridable native function for when this controller is asked to possess a pawn.
* @param InPawn The Pawn to be possessed
*/
void AAmsPlayerController::OnPossess(APawn * aPawn)
{
	Super::OnPossess(aPawn);
	// LOG_ADV_PNMS( "Possessing current Pawn [%s]", *GetNameSafe(aPawn));

	// If the current controller is set to be in "Auto Move" mode, then do
	if ( !this->bIsAutoMovementMode )
	{
		this->riderControllerComponent->SetControlledPawn( aPawn );
	}
	else
	{
		this->bIsAutoMovementMode = false;

		//FTimerHandle timerHandle;
		//this->GetWorld()->GetTimerManager().SetTimer(timerHandle, this, &AAmsPlayerController::OnPawnFinishedMovingToLocation, 0.25, false);
		//this->OnPawnFinishedMovingToLocation();
		
	}
}

/**
* Called when the pawn is unpossessed
* @Todo: Remove Function if not useful
*/
void AAmsPlayerController::OnUnPossess()
{	
	//LOG_ADV_PNMS( "Unpossessing current Pawn");
	Super::OnUnPossess();

	// Do not call super here, this has been copied from the AController::OnUnPossess Logic
	// to remove the view target swap logic.
	/*if ( GetPawn() != NULL )
	{
		if ( GetLocalRole() == ROLE_Authority )
		{
			GetPawn()->SetReplicates( true );
		}
		GetPawn()->UnPossessed();

		if ( GetViewTarget() == GetPawn() )
		{
			SetViewTarget( this );
		}
	}
	SetPawn( NULL );*/
}

/** Called on the client to do local pawn setup after possession, before calling ServerAcknowledgePossession */
void AAmsPlayerController::AcknowledgePossession(APawn * pawn)
{
	Super::AcknowledgePossession(pawn);
	//LOG_ADV_PNMS("Acknowledging Possession of the new pawn [%s]", *GetNameSafe(pawn));

	// In Multilayer, initialize or reinitialize the ability system.
	IAbilitySystemInterface * absPawn = Cast<IAbilitySystemInterface>(pawn);
	if(absPawn)
	{
		absPawn->GetAbilitySystemComponent()->InitAbilityActorInfo(pawn, pawn);
	}

	// If the controller is currently inAutoMove Mode, we can Finish Mounting to Location.
	if ( this->bIsAutoMovementMode )
	{
		this->bIsAutoMovementMode = false;
	}
}

/**
* Called only on clients to set the auto move to enabled or disabled from the pawn
*/
void AAmsPlayerController::ClientSetAutoMoveMode( bool isAutoMove )
{
	if ( this->GetNetMode() != ENetMode::NM_Client )
	{
		return;
	}

	this->bIsAutoMovementMode = isAutoMove;
}

/**
* Switch the current pawn of this player to use the AI Controller
*/
void AAmsPlayerController::SwitchPawnToAiController( bool bIsClientRequest)
{
	this->bIsAutoMovementMode = true;

	if (!this->HasAuthority() )
	{
		this->SERVER_SwitchPawnToAiController();
		return;
	}

	if ( !IsValid( this->aiPlayerController ) )
	{
		if ( bIsClientRequest )
		{
			this->CLIENT_ResetAutoMoveFlag(false);
		}

		return;
	}

	this->aiPlayerController->Possess(this->riderControllerComponent->GetOwnedPawn());
}

/**
* Switch the current pawn of this player to use the player controller
*/
void AAmsPlayerController::SwitchPawnToPlayerController( bool bIsClientRequest)
{
	if ( !this->HasAuthority() )
	{
		//this->SERVER_SwitchPawnToPlayerController();
		return;
	}

	// something really bad happened and for some reason we no longer have an owned pawn
	APawn * const ownedPawn = this->riderControllerComponent->GetOwnedPawn();
	if ( !IsValid( ownedPawn ) )
	{
		if ( bIsClientRequest )
		{
			this->CLIENT_ResetAutoMoveFlag(true);
		}
		return;
	}

	this->Possess( this->riderControllerComponent->GetOwnedPawn());
}

/**
* Move the player to the specified location
*/
void AAmsPlayerController::AutoMovePlayerToLocation( FVector const & location )
{
	this->bIsAutoMovementMode = true;
	if ( !this->HasAuthority() )
	{
		this->SERVER_AutoMovePlayerToLocation(location);
		return;
	}

	this->aiPlayerController->SetTargetLocation(location);
	this->SwitchPawnToAiController();
}

/**
* Spawns the player AI Controller to be used for movement later
*/
void AAmsPlayerController::SpawnPlayerAIController()
{
	if (!this->HasAuthority() )
	{
		return;
	}

	UWorld * const world = this->GetWorld();
	if (!IsValid( world ) )
	{
		return;
	}

	FActorSpawnParameters params;
	params.Owner = this;
	params.SpawnCollisionHandlingOverride = ESpawnActorCollisionHandlingMethod::AlwaysSpawn;
	
	this->aiPlayerController = world->SpawnActor<AAmsAIPlayerController>(this->defaultAiPlayerControllerClass, params);
	this->aiPlayerController->SetOwningPlayerController(this);
}

/**
* Destroys the player AI Controller
*/
void AAmsPlayerController::DestroyPlayerAIController()
{
	if ( !this->HasAuthority() )
	{
		return;
	}

	if ( !IsValid( this->aiPlayerController ) )
	{
		return;
	}

	this->aiPlayerController->Destroy();
	this->aiPlayerController = nullptr;
}

// Delegate event fired when the pawn has changed
void AAmsPlayerController::OnPossessedPawnChangedEvent( APawn * oldPawn, APawn * newPawn )
{
	if ( !IsValid( this->riderControllerComponent ) )
	{
		return;
	}

	APawn * const ownedPawn = this->riderControllerComponent->GetOwnedPawn();
	if ( !IsValid( ownedPawn ) )
	{
		return;
	}

	if ( newPawn == nullptr)
	{
		this->SetViewTarget(ownedPawn);
	}
}

// IAdmUnifiedControllerPawn
////////////////////////////////////////////////////////////////////////////

/** 
* Get the Character's Controller
* This will only work on the owning client or the server
* @return class AController *
*/
class AController * AAmsPlayerController::GetCharacterController_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}
	return Cast<AController>(this->riderControllerComponent->GetOwner());
}

/** 
* Get the Character that the player or bot is represented as
* @return class APawn *
*/
class APawn * AAmsPlayerController::GetCharacterPawn_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}

	return this->riderControllerComponent->GetOwnedPawn();
}

/**
* Get the Mount that the character is riding, null if no mount.
* @return class APawn *
*/
class AActor * AAmsPlayerController::GetCharacterMount_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}

	return this->riderControllerComponent->GetMountActor();
}

/** 
* Get flag indicating that the character is mounted.
* this works for either the Rider or the Mount itself
*/
bool AAmsPlayerController::IsMounted_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return false;
	}

	return this->riderControllerComponent->IsMounted();
}

/**
* Called to Prepare the Pawn or Controller for Mounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsPlayerController::PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response)
{
	return true;
}

/**
* Prepare the Pawn or Controller for Dismounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsPlayerController::PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response)
{
	return true;
}

// IRiderControllerInterface
////////////////////////////////////////////////////////////////////////////

/**
* Begins the Mounting Process
* @param newMount - Actor to mount to
* @param linkedActor - Linked Actor to mount to
* @param seatId - seat to mount to
* @param mountingResponse - Status Response of the mounting if it succeeded.
* @return bool - true if the mounting operation succeeded.
*/
bool AAmsPlayerController::BeginMountingActor_Implementation(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformMountActor(newMount, linkedActor, mountingResponse);
}

/**
* Begins the Mounting Process
* @param newMount - Actor to mount to
* @param linkedActor - Linked Actor to mount to
* @param seatId - seat to mount to
* @param mountingResponse - Status Response of the mounting if it succeeded.
* @return bool - true if the mounting operation succeeded.
*/
bool AAmsPlayerController::BeginMountingActorToSeat_Implementation(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformMountActorToSeat(newMount, linkedActor, seatId, mountingResponse);
}

/** 
* Starts the dismounting action
* @param response - output status response if mounting succeeded or failed
* @return bool - true if dismount succeeded.
*/
bool AAmsPlayerController::BeginDismountingActor_Implementation(FMountActionResponse & response)
{
	AActor * const mountActor = this->riderControllerComponent->GetMountActor();
	return this->riderControllerComponent->PerformDismountActor(mountActor, response);
}

/**
* Begin changing seats to the specified seat ID.
* @param seatId - ID of the seat to change to
* @param mountingResponse - status Response of the seat change
* @return bool - true if the seat change was successful
*/
bool AAmsPlayerController::BeginChangingSeatById_Implementation(int32 seatId, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformChangeSeatToIndex(seatId, mountingResponse);
}

/**
* Begin changing seats to the specified seat index.
* @param seatIndex - index of the seat to change to
* @param mountingResponse - status Response of the seat change
* @return bool - true if the seat change was successful
*/
bool AAmsPlayerController::BeginChangingSeatToIndex_Implementation(int32 seatIndex, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformChangeSeatToIndex(seatIndex, mountingResponse);
}

/** 
* Get the Rider Controller Component
*/
class URiderControllerComponent * AAmsPlayerController::GetRiderControllerComponent_Implementation() const
{
	return this->riderControllerComponent;
}

/** 
* Called to tell the controller to possess the given Pawn
*/
bool AAmsPlayerController::PNMS_PossessPawn_Implementation(APawn * pawnToPossess)
{
	 return this->riderControllerComponent->PossessPawn(pawnToPossess);
}

/** 
* Called when a new rider is added to the current mount
*/
bool AAmsPlayerController::OnRiderAdded_Implementation(AActor * newRider, int32 seatId)
{
	this->riderControllerComponent->RiderAdded(newRider, seatId);
	return true;
}

/**
* Called when a rider is removed from the current mount
*/
bool AAmsPlayerController::OnRiderRemoved_Implementation(AActor * removedRider, int32 seatId)
{
	this->riderControllerComponent->RiderRemoved(removedRider, seatId);
	return true;
}

/**
* Called when another rider changes to a new seat on the mount
*/
bool AAmsPlayerController::OnOtherRiderChangedSeats_Implementation(AActor * otherRider, int32 newSeatId, int32 oldSeatId)
{
	this->riderControllerComponent->OtherRiderChangedSeat(otherRider, newSeatId, oldSeatId);
	return true;
}

/**
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to teh failure of the mounting action.
*/
bool AAmsPlayerController::OnPnmsMountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs)
{
	this->riderControllerComponent->ResetToDismounted(pnmsActionResultArgs);
	return true;
}

/**
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
*/
bool AAmsPlayerController::OnPnmsDisountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs)
{
	this->riderControllerComponent->ResetToMounted(pnmsActionResultArgs);

	return true;
}

/**.
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
*/
bool AAmsPlayerController::OnPnmsChangeSeatActionFailed_Implementation(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs)
{
	return true;
}

// Network and RPCs
////////////////////////////////////////////////////////////////////////////

/**
* Setup the replicated properties for this class
*/
void AAmsPlayerController::GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const
{
	Super::GetLifetimeReplicatedProps(OutLifetimeProps);

	DOREPLIFETIME(ThisClass, riderControllerComponent);
}

/**
* Called to switch pawn to AI Controller
*/
void AAmsPlayerController::SERVER_SwitchPawnToAiController_Implementation()
{
	this->SwitchPawnToAiController(true);
}

bool AAmsPlayerController::SERVER_SwitchPawnToAiController_Validate()
{
	return true;
}

/**
* Called to switch the pawn to the Player Controller
*/
void AAmsPlayerController::SERVER_SwitchPawnToPlayerController_Implementation()
{
	this->SwitchPawnToPlayerController(true);
}

bool AAmsPlayerController::SERVER_SwitchPawnToPlayerController_Validate()
{
	return true;
}

// Called to start the auto move system on the server
void AAmsPlayerController::SERVER_AutoMovePlayerToLocation_Implementation( FVector const & location )
{
	this->AutoMovePlayerToLocation(location);
}

bool AAmsPlayerController::SERVER_AutoMovePlayerToLocation_Validate( FVector const & location )
{
	return true;
}

// In the event there is an error on the server when enabling and disabling automatic movement, inform the client to reset
void AAmsPlayerController::CLIENT_ResetAutoMoveFlag_Implementation( bool bAutoMoveFlag )
{
	this->bIsAutoMovementMode = bAutoMoveFlag;
	
	/*AAmsRiderCharacter * const pawn = Cast<AAmsRiderCharacter>(this->riderControllerComponent->GetOwnedPawn());
	if ( IsValid( pawn ) )
	{
		pawn->OnMoveToMountingLocationCompleted();
	}*/
}
