// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#include "Pawn/AmsRiderCharacter.h"
#include "Components/MountRiderComponent.h"
#include "Player/AmsPlayerController.h"
#include "PnmsAdvancedSystemPublicPCH.h"

// AMS Includes
#include "Interfaces/IAmsMountableActor.h"

// UE Includes
#include "Runtime/Engine/Classes/Components/CapsuleComponent.h"
#include "AbilitySystemComponent.h"
#include "Runtime/Engine/Classes/GameFramework/CharacterMovementComponent.h"
#include "Runtime/Engine/Classes/Animation/AnimMontage.h"
#include "Runtime/AIModule/Classes/AIController.h"
#include "Runtime/Engine/Public/TimerManager.h"

// Basic Configuration
///////////////////////////////////////////////////////////////////////////////////
	
/**
* Sets default values for this character's properties
*/
AAmsRiderCharacter::AAmsRiderCharacter()
{
 	// Set this character to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	// Properties
	this->mountRiderComponent = CreateDefaultSubobject<UMountRiderComponent>(TEXT("MountRiderComponent"));

	this->abilitySystemComponent = CreateDefaultSubobject<UAbilitySystemComponent>(TEXT("AbilitySystemComponent"));
	this->abilitySystemComponent->SetIsReplicated(true);
}

/**
* Called when the character is possessed by a new Controller on the server
*/
void AAmsRiderCharacter::PossessedBy(AController * newController)
{
	Super::PossessedBy(newController);
	//LOG_ADV_PNMS("New Controller [%s] possessing pawn", *newController->GetHumanReadableName());

	// Initialize or Reinitialize the Ability System Component
	if(IsValid(this->abilitySystemComponent))
	{
		this->abilitySystemComponent->InitAbilityActorInfo(this, this);
	}

	this->SetOwner(newController);
}

// Methods
///////////////////////////////////////////////////////////////////////////////////

/**
* Called to begin the small wait timer for the switch over to the player controller
*/
void AAmsRiderCharacter::StartMountingLocaitonCompleteTimer(float duration /*= 0.5f*/)
{
	UWorld * const world = this->GetWorld();
	if(IsValid(world))
	{
		FTimerHandle timerHandle;
		world->GetTimerManager().SetTimer(timerHandle, this, &AAmsRiderCharacter::OnMoveToLocationCompleted, duration, false);
	}
}

/**
* Called to begin the small wait timer for the switch over to the player controller
*/
void AAmsRiderCharacter::OnMoveToLocationCompleted()
{
	this->mountRiderComponent->MoveToMountingLocationComplete();
}

// IAdmUnifiedControllerPawn
////////////////////////////////////////////////////////////////////////////

/** 
* Get the Character's Controller
* This will only work on the owning client or the server
* @return class AController *
*/ 
class AController * AAmsRiderCharacter::GetCharacterController_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return nullptr;
	}

	return this->mountRiderComponent->GetRiderController();
}

/** 
* Get the Character that the player or bot is represented as
* @return class APawn *
*/
class APawn * AAmsRiderCharacter::GetCharacterPawn_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return nullptr;
	}

	return Cast<APawn>(this->mountRiderComponent->GetOwner());
}

/**
* Get the Mount that the character is riding, null if no mount.
* @return class APawn *
*/
class AActor * AAmsRiderCharacter::GetCharacterMount_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return nullptr;
	}

	return this->mountRiderComponent->GetCurrentMount();
}

/** 
* Get flag indicating that the character is mounted.
* this works for either the Rider or the Mount itself
*/
bool AAmsRiderCharacter::IsMounted_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->IsMounted();
}

/**
* Called to Prepare the Pawn or Controller for Mounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsRiderCharacter::PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response)
{
	return true;
}

/**
* Prepare the Pawn or Controller for Dismounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsRiderCharacter::PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response)
{
	return true;
}

// IMountRiderInterface
///////////////////////////////////////////////////////////////////////////////////

/**
* Get the Mount Rider Component
*/
class UMountRiderComponent * AAmsRiderCharacter::GetMountRiderComponent_Implementation() const
{
	return this->mountRiderComponent;
}

/**
* Starts mounting process for given mounted pawn
* @param newMountActor - Actor to mount
* @param linkedActor - optional actor with a link to the mount
* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the mounting
* @version 4
*/
bool AAmsRiderCharacter::StartPawnMounting_Implementation(class AActor * newMountActor, AActor * linkedActor, struct FMountActionResponse & mountingResponse)
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->MountPawn(newMountActor, linkedActor, mountingResponse);
}

/**
* Starts mounting process for given mounted pawn
* @param newMountActor - Actor to mount
* @param linkedActor - optional actor with a link to the mount
* @param seatId - seat to attach to
* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the mounting
* @version 4
*/
bool AAmsRiderCharacter::StartPawnMountingToSeat_Implementation(class AActor * newMountActor, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse)
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->MountPawn(newMountActor, linkedActor, mountingResponse);
}

/**
* Called when the mounting process has completed
* @param newMountActor - Actor to mount
* @param linkedActor - optional actor with a link to the mount
* @return bool - true in order to treat as function in blueprint instead of event
*/
bool AAmsRiderCharacter::OnMountingPawnFinished_Implementation(class AActor * newMountActor)
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->PawnFinishedMounting();
}

/**
* Starts dismounting process for given mounted pawn
* @param oldPawnMount - Actor to dismount
* @return bool - this can be leveraged to indicate that the Rider Needs to run an animation in order to complete the dismounting
* @version 4
*/
bool AAmsRiderCharacter::StartPawnDismounting_Implementation(class AActor * oldPawnMount, struct FMountActionResponse & mountingResponse)
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->DismountPawn(oldPawnMount, mountingResponse);
}

/**
* Called when the mounting process has completed
* @param dismountedPawn - APawn being dismounted
* @return bool - true in order to treat as function in blueprint instead of event
*/
bool AAmsRiderCharacter::OnDismountingPawnFinished_Implementation(class AActor * oldPawnMount)
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->PawnFinishedDismounting();
}

/**
* Get flag indicating that the rider is the driver of the mount/vehicle
* @return bool
*/
bool AAmsRiderCharacter::IsDriver_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->IsDriver();
}

/**
* Get flag indicating that the rider is seated on the mount.
* @return bool
*/
bool AAmsRiderCharacter::IsSeatedOnMount_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->IsSeated();
}

/** 
* Get the ID of the seat the rider is currently mounted on.
*/
int32 AAmsRiderCharacter::GetSeatId_Implementation() const
{
	if(!IsValid(this->mountRiderComponent))
	{
		return -1;
	}

	return this->mountRiderComponent->GetSeatId();
}

/**
* Allow movement to the specified location
* @return bool - true if teleporting to that location, false if movement to location is required
*/
bool AAmsRiderCharacter::MoveToMountingLocation_Implementation(const FVector & location, const FRotator & orientation)
{
	//LOG_ADV_PNMS("Move to Mounting Location Called");

	// For the Auto movement to work the character needs to at a range where it can move into locaiton.
	// if it is too close then the system will bug out because the AI Controller possession, move, and then unpossession process is too quick
	// and the client does not have enough time to adjust.
	FVector pawnLocation = this->GetActorLocation();
	FVector targetLocation = location;
	pawnLocation.Z = 0.0f;
	targetLocation.Z = 0.0f;

	FVector const targetDistance = targetLocation - pawnLocation;
	float const maxDistanceSquared = this->minMountDistanceMoveThreshold * this->minMountDistanceMoveThreshold;
	float const distanceSquared = targetDistance.SizeSquared();

	if(distanceSquared <= maxDistanceSquared)
	{
		// return false that we are not going to move.
		return false;
	}

	// If we are far enough away from the mount then we can invoke the auto movement system to move us into position.
	ENetMode netMode = this->GetNetMode();
	AController * const controller = this->mountRiderComponent->GetRiderController();

	if(!IsValid(controller))
	{
		return true;
	}

	if(controller->IsPlayerController())
	{
		auto * const pc = Cast<AAmsPlayerController>(this->mountRiderComponent->GetRiderController());
		// If controller is not valid then we are most likely dealing with an AI Controller.
		if(!IsValid(pc))
		{
			return false;
		}

		// we need to make some considerations for client side, if this is a local controller it needs to sync its move with the server.
		if(netMode == NM_Client)
		{
			// If the controller is valid on a client then it must be a local controller, set it to auto move
			if(IsValid(pc))
			{
				pc->ClientSetAutoMoveMode(true);
			}

			return true;
		}

		// Only perform a move if we are on the authority
		if(this->HasAuthority())
		{
			pc->AutoMovePlayerToLocation(location);
		}
	}
	else
	{
		auto * const aiController = Cast<AAIController>(controller);
		if(IsValid(aiController))
		{
			return false;
		}

		aiController->MoveToLocation(location, 50.0f);
	}

	return true;
}

/**
* Called when moving the the desired mounting location is completed
*/
bool AAmsRiderCharacter::OnMoveToMountingLocationCompleted_Implementation()
{
	LOG_ADV_PNMS( "Move to Mounting Location Completed" );
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->MoveToMountingLocationComplete();
}

/**
* Called when player has finished changing to their new seat
*/
bool AAmsRiderCharacter::OnChangeToNewSeatCompleted_Implementation()
{
	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	return this->mountRiderComponent->PawnFinishedChangingSeats();
}

/**
* Play mounting animation based on current position
*/
bool AAmsRiderCharacter::PlayMountingAnimation_Implementation(EMountingDirection position)
{
	if(!IsValid(this->mountingAnimationData))
	{
		return false;
	}

	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	AActor * const mountActor = this->mountRiderComponent->GetCurrentMount();
	if(!IsValid(mountActor))
	{
		return false;
	}

	if(!IAmsMountableActorInterface::ImplementsInterface(mountActor))
	{
		return false;
	}

	EAmsMountTypes const mountType = IAmsMountableActorInterface::Execute_GetMountType(mountActor);
	UAnimMontage * const mountingMontage = this->mountingAnimationData->GetMontageForMounting(mountType, position);
	if(!IsValid(mountingMontage))
	{
		return false;
	}

	this->PlayAnimMontage(mountingMontage);
	return true;
}

/**
* Play mounting animation based on current position
*/
bool AAmsRiderCharacter::PlayDismountingAnimation_Implementation(EMountingDirection position)
{
	if(!IsValid(this->mountingAnimationData))
	{
		return false;
	}

	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	AActor * const mountActor = this->mountRiderComponent->GetPreviousMount();
	if(!IsValid(mountActor))
	{
		return false;
	}

	if(!IAmsMountableActorInterface::ImplementsInterface(mountActor))
	{
		return false;
	}

	EAmsMountTypes const mountType = IAmsMountableActorInterface::Execute_GetMountType(mountActor);
	UAnimMontage * const dismountingMontage = this->mountingAnimationData->GetMontageForDismounting(mountType, position);
	if(!IsValid(dismountingMontage))
	{
		return false;
	}

	this->PlayAnimMontage(dismountingMontage);
	return true;
}

/**
* Play animation for moving character into their new seat.
*/
bool AAmsRiderCharacter::PlayMoveToSeatAnimation_Implementation(int32 currentSeatId, int32 oldSeatId)
{
	if(!IsValid(this->mountingAnimationData))
	{
		return false;
	}

	if(!IsValid(this->mountRiderComponent))
	{
		return false;
	}

	AActor * const mountActor = this->mountRiderComponent->GetCurrentMount();
	if(!IsValid(mountActor))
	{
		return false;
	}

	if(!IAmsMountableActorInterface::ImplementsInterface(mountActor))
	{
		return false;
	}

	EAmsMountTypes const mountType = IAmsMountableActorInterface::Execute_GetMountType(mountActor);
	UAnimMontage * const changeSeatMontage = this->mountingAnimationData->GetMontageForChangingSeatsById(mountType, oldSeatId, currentSeatId);
	if(!IsValid(changeSeatMontage))
	{
		return false;
	}

	this->PlayAnimMontage(changeSeatMontage);
	return true;
}

/**
* Get the Mesh component of the Rider
*/
UMeshComponent * AAmsRiderCharacter::GetRiderMesh_Implementation() const
{
	return this->GetMesh();
}

/**
* Enable or Disable the collision of a Rider
* @param collisionType - new collision type for the rider
* @return bool - true if collision changes where handled by this function
* @Version 6
*/
bool AAmsRiderCharacter::SetRiderCollisionEnabled_Implementation(bool shouldEnable)
{
	if(shouldEnable)
	{
		this->GetCapsuleComponent()->SetCollisionEnabled(ECollisionEnabled::QueryAndPhysics);
	}
	else
	{
		this->GetCapsuleComponent()->SetCollisionEnabled(ECollisionEnabled::NoCollision);
	}

	return true;
}

/**
* Updates the rider movement mode for when the character is mounted or not
* @param isMounted - bool flag to indicate if we are mounted or not
* @param mountActor - actor that is currently our mount.
* @return bool - true if we handled the update.
*/
bool AAmsRiderCharacter::UpdateRiderMovementMode_Implementation(const bool isMounted, const AActor * mountActor)
{
	UCharacterMovementComponent * const movementComp = this->GetCharacterMovement();
	if(!IsValid(movementComp))
	{
		return false;
	}

	// Set the Movement Mode for the character to its semi-default state
	if(isMounted)
	{
		movementComp->SetMovementMode(EMovementMode::MOVE_Walking);
	}
	else
	{
		movementComp->SetMovementMode(EMovementMode::MOVE_None);
	}

	// if mounted, then ignore client corrections for THIS object.
	movementComp->bIgnoreClientMovementErrorChecksAndCorrection = isMounted;

	return true;
}

/**
* Retrieve all Skeletal Mesh Objects that make up a Rider which need to Animate on a Server.
* This is a bug fix for an issue which occurs when turning Replicate Movement off on both Listen and Dedicated Servers
* which stops all animation updates on the server.
* @return TArray<USkeletalMeshComponent*>
*/
TArray<class USkeletalMeshComponent*> AAmsRiderCharacter::GetAllRiderSkeletalMeshes_Implementation() const
{
	TArray<USkeletalMeshComponent *> skeletalMesheList;
	skeletalMesheList.Add(this->GetMesh());

	return skeletalMesheList;
}

// IAmsMountRider Interface
///////////////////////////////////////////////////////////////////////////////////

/**
* Gets the Mounting Animation Data Objiect for the Rider
* @return UAmsMountingAnimationData *
*/
class UAmsMountingAnimationData * AAmsRiderCharacter::GetMountingAnimationData_Implementation() const
{
	if(!IsValid(this->mountingAnimationData))
	{
		return nullptr;
	}

	return this->mountingAnimationData;
}

/**
* Find the AnimMontage for mounting based on the mount type, the mounting direction
* @param mountType - type of mount being used
* @param mountingDirection - direction that the mounting is occuring
* @return UAnimMontage to play the animation.
*/
class UAnimMontage * AAmsRiderCharacter::GetMontageForMounting_Implementation(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const
{
	if(!IsValid(this->mountingAnimationData))
	{
		return nullptr;
	}

	return this->mountingAnimationData->GetMontageForMounting(mountType, mountingDirection);
}

/**
* Find the AnimMontage for dismounting based on the mount type, the mounting direction
* @param mountType - type of mount being used
* @param mountingDirection - direction that the mounting is occuring
* @return UAnimMontage to play the animation.
*/
class UAnimMontage * AAmsRiderCharacter::GetMontageForDismounting_Implementation(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const
{
	if(!IsValid(this->mountingAnimationData))
	{
		return nullptr;
	}

	return this->mountingAnimationData->GetMontageForDismounting(mountType, mountingDirection);
}

/**
* Get the AnimMontage for changing from one seat to another.
* @param mountType - type of mount being used
* @param startSeatName - name of the seat the change is starting from
* @param destinationSeatName - name of the seat to end at
* @return UAnimMontage to play
*/
class UAnimMontage * AAmsRiderCharacter::GetMontageForChangingSeatsByName_Implementation(EAmsMountTypes const mountType, FName const & startingSeatName, FName const & destinationSeatName) const
{
	if(!IsValid(this->mountingAnimationData))
	{
		return nullptr;
	}

	return this->mountingAnimationData->GetMontageForChangingSeats(mountType, startingSeatName, destinationSeatName);
}

/**
* Get the AnimMontage for changing from one seat to another.
* @param mountType - type of mount being used
* @param startSeatName - name of the seat the change is starting from
* @param destinationSeatName - name of the seat to end at
* @return UAnimMontage to play
*/
class UAnimMontage * AAmsRiderCharacter::GetMontageForChangingSeatsById_Implementation(EAmsMountTypes const mountType, int32 const startingSeatId, int32 const destinationSeatId) const
{
	if(!IsValid(this->mountingAnimationData))
	{
		return nullptr;
	}

	return this->mountingAnimationData->GetMontageForChangingSeatsById(mountType, startingSeatId, destinationSeatId);
}

/**
* Enables and Disabled the auto move functionality on the rider
*/
bool AAmsRiderCharacter::SetAutoMoveEnabled_Implementation(bool isEnabled)
{
	if(this->GetLocalRole() >= ROLE_Authority)
	{
		bool wasEnabled = this->isAutomoveEnabled;
		this->isAutomoveEnabled = isEnabled;

		if(wasEnabled && !this->isAutomoveEnabled)
		{
			this->StartMountingLocaitonCompleteTimer(0.25);
		}
	}

	return isEnabled;
}

// IAbilitySystemInterface
///////////////////////////////////////////////////////////////////////////////////

/** Returns the ability system component to use for this actor. It may live on another actor, such as a Pawn using the PlayerState's component */
UAbilitySystemComponent* AAmsRiderCharacter::GetAbilitySystemComponent() const
{
	return this->abilitySystemComponent;
}

// Network and RPCs
////////////////////////////////////////////////////////////////////////////
/**
* Setup the replicated properties for this class
*/
void AAmsRiderCharacter::GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const
{
	Super::GetLifetimeReplicatedProps(OutLifetimeProps);

	DOREPLIFETIME(ThisClass, mountRiderComponent);
	DOREPLIFETIME(ThisClass, isAutomoveEnabled);
}

/**
* Called when auto move enable and disable is replicated
*/
void AAmsRiderCharacter::OnRep_IsAutomoveEnabled(bool wasAutomoveEnabled)
{
	// if this value replicates and is identical then do nothing.
	if(this->isAutomoveEnabled == wasAutomoveEnabled)
	{
		return;
	}

	// if automove was enabled and now it is not, then the move has completed.
	if(!isAutomoveEnabled && wasAutomoveEnabled)
	{
		this->StartMountingLocaitonCompleteTimer(0.25);
	}
}
